/* Copyright (C) 2014-2018 RealVNC Ltd. All Rights Reserved.
 * */

#ifndef __VNCDISCOVERYSDKUTILS_H__
#define __VNCDISCOVERYSDKUTILS_H__

#include <vncdiscoverysdktypes.h>

/**
* \file vncdiscoverysdktypes.h
*
* This is the header file that some utility methods that can be used by the
* client application, Discoverers and Device Providers.
*
* Methods defined here provide mechanisms for generating strings which can be
* used for example in the fetch/post entity value calls. The strings would
* contain information normally found in structures.
*
* Similarly, there are methods for parsing the strings which have been
* generated.
*
* This allows, for example, a Discoverer to respond to a Fetch Entity Value with
* more complex information. If the information was:
*  {
*    field1: value1;
*    field2: value2;
*    field3: value3;
*    complexField: {
*      subField1: value1;
*      subField2: value2;
*    }
*    arrayField: {
*     elemnt1, element2, element3;
*    }
*  }
*
* Then the generator will encode this into a string, which is returned to the
* client application. The client application can then use the parser to find the
* values of field1, field2, or or any other fields.
*
* \section using_the_generator Using the Generator
*
* The Discoverer would need to call first. VNCDiscoverySDKUtilCreateGenerator().
* Then it would need to set the fields by calling:
*   VNCDiscoverySDKGeneratorSetStringField(generator, "field1", "value1")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "field2", "value2")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "field3", "value3")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "complexField.subField1", "value1")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "complexField.subField2", "value2")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "arrayField.0", "element1")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "arrayField.1", "element2")
*   VNCDiscoverySDKGeneratorSetStringField(generator, "arrayField.2", "element3")
*
* Finally, it would generate the string by calling:
*   VNCDiscoverySDKGeneratorGenerate(generator)
*
* When no longer needed, the Discoverer must call:
*   VNCDiscoverySDKGeneratorDestroy(generator)
*
* \section using_the_parser Using the Parser
*
* When the client application receives the string, it can call:
*   VNCDiscoverySDKUtilCreateParser(string)
*
* If the parser was created correctly, then the application can retrieve any
* field through:
*   VNCDiscoverySDKParserGetStringField(parser, "complexField.subField2")
*
* If the application wishes to iterate through the array it must call:
*   VNCDiscoverySDKParserGetSubFieldCount(parser, "arrayField")
* and then it can retrieve the elements:
*   VNCDiscoverySDKParserGetStringField(parser, "arrayField.0")
*
* When the parser is no longer needed, the application can destroy it:
*   VNCDiscoverySDKParserDestroy(parser)
*
* \section structure_representation Generator and Parser Structure Representation
*
* The fields, for both the Generator and Parser, can be laid out in a
* hierarchy. Each field can be either containing other fields, or a leaf filed.
* Only leaf fields are allowed to have a value.
*
* The path to a leaf field is represented as a string composed of the fields
* that need to be visited to reach the leaf field. The fields from the path are
* separated by a dot. See the examples from the Generator and Parser sections.
*
* There is no strict definition of an array, but if an array is to be defined,
* the convention is that this can be a non-leaf field which has children with
* the name "0", "1", "2", up to array_count - 1.
*
* \see VNCDiscoverySDKGetUtils
* \see VNCDiscovererGetUtils
* \see VNCDeviceProviderGetUtils
* \see VNCDiscoverySDKGenerator
* \see VNCDiscoverySDKParser
*/

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Creates an instance of the Generator.
 *
 * The Generator can then be used to add information that can be compiled into a
 * string.
 *
 * \param pSdk The SDK instance. This can be retrieved from the VNCDiscoverySDKUtils
 * structure.
 * \param ppGenerator The location where the Generator will be placed if successfully
 * created.
 *
 * \retval VNCDiscoverySDKErrorNone The Generator was created successfully.
 * \retval VNCDiscoverySDKErrorOutOfMemory There is not enough memory to
 *         initialise the Generator.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL.
 *
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKGeneratorSetStringField
 * \see VNCDiscoverySDKGeneratorSetInt32Field
 * \see VNCDiscoverySDKGeneratorGenerate
 * \see VNCDiscoverySDKGeneratorDestroy
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKUtilCreateGenerator(VNCDiscoverySDKInstance *pSdk,
    VNCDiscoverySDKGenerator **ppGenerator);

/**
 * \brief Sets one string field in the Generator.
 *
 * The field set must be a leaf field, as they are the only ones that are
 * allowed a value. The path will be automatically created by the Generator. If
 * a field along the path already has a value, then this will fail, as it means
 * that the field is supposed to be a leaf-field.
 *
 * A field could be re-set. The new set just overrides the previous one.
 *
 * \param pGenerator Pointer to the Generator where the field should be set.
 * \param fieldPath The path to the field that should receive the value. Must
 * not be NULL. The allowed values for a value are letters (a-z and A-Z),
 * numbers (0-9) and underscore (_). The fields are separated by a dot (.).
 * \param value The value to which to set the field. This is a NUL-terminated
 * UTF-8 encoded string. Must not be NULL. Ownership stays with the caller.
 *
 * \retval VNCDiscoverySDKErrorNone The field was set successfully.
 * \retval VNCDiscoverySDKErrorOutOfMemory The field can not be set due to
 * not enough memory.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL,
 * or the fieldPath is not in the expected format.
 * \retval VNCDiscoverySDKFailed The path contains leaf fields, or the target
 * field is already a non-leaf field.
 *
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKUtilCreateGenerator
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKGeneratorSetStringField(VNCDiscoverySDKGenerator *pGenerator,
    const char *fieldPath, const char *value);

/**
 * \brief Sets one signed 32 bit integer field in the Generator.
 *
 * The field set must be a leaf field, as they are the only ones that are
 * allowed a value. The path will be automatically created by the Generator. If
 * a field along the path already has a value, then this will fail, as it means
 * that the field is supposed to be a leaf-field.
 *
 * A field could be re-set. The new set just overrides the previous one.
 *
 * \param pGenerator Pointer to the Generator where the field should be set.
 * \param fieldPath The path to the field that should receive the value. Must
 * not be NULL. The allowed values for a value are letters (a-z and A-Z),
 * numbers (0-9) and underscore (_). The fields are separated by a dot (.).
 * \param value The value to which to set the field.
 *
 * \retval VNCDiscoverySDKErrorNone The field was set successfully.
 * \retval VNCDiscoverySDKErrorOutOfMemory The field can not be set due to not
 * enough memory.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL,
 * or the fieldPath is not in the expected format.
 * \retval VNCDiscoverySDKFailed The path contains leaf fields, or the target
 * field is already a non-leaf field.
 *
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKUtilCreateGenerator
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKGeneratorSetInt32Field(VNCDiscoverySDKGenerator *pGenerator,
    const char *fieldPath, vnc_int32_t value);

/**
 * \brief Generates the string that corresponds to the structure set within the
 * Generator.
 *
 * Once all the fields have been set, this should be called to convert
 * everything into a single string which can be passed to the Discovery SDK.
 *
 * \param pGenerator Pointer to the Generator that should create the string.
 * \param ppResult The location where the result will be stored, if the
 * Generator is successful. Ownership of the string stays with the Generator.
 * The lifetime of the string is tied to the lifetime of the Generator. The
 * string is NUL terminated. The string can be used to be passed around in SDK
 * requests, but its contents must not be changed. It must be read only by using
 * the Parser to do so. If the Generator fails, then this will be NULL.
 *
 * \retval VNCDiscoverySDKErrorNone The string was generated successfully.
 * \retval VNCDiscoverySDKErrorOutOfMemory The string can not be generated due
 * to not enough memory.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL,
 * or the fieldPath is not in the expected format.
 *
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKUtilCreateGenerator
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKGeneratorGenerate(VNCDiscoverySDKGenerator *pGenerator,
    const char **ppResult);

/**
 * \brief Destroys the Generator.
 *
 * Once the Generator is no longer needed, this must be called to free up any
 * resources.
 *
 * \param pGenerator The Generator that should be destroyed.
 *
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKUtilCreateGenerator
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef void VNCCALL
VNCDiscoverySDKGeneratorDestroy(VNCDiscoverySDKGenerator *pGenerator);

/**
 * \brief Creates an instance of the Parser.
 *
 * The Parser can then be used to retrieve information from a parsed string.
 *
 * One parser per string is created. If you need to parse a new string, then
 * create a new parser.
 *
 * \param pSdk The SDK instance. This can be retrieved from the VNCDiscoverySDKUtils
 * structure.
 * \param string The string for which to create the Parser. This is a string
 * generated through the Generator. Must not be NULL. Ownership stays with the
 * caller.
 * \param ppParser The location where the Parser will be placed if successfully
 * created.
 *
 * \retval VNCDiscoverySDKErrorNone The Parser was created successfully.
 * \retval VNCDiscoverySDKErrorOutOfMemory There is not enough memory to
 *         initialise the Parser.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL.
 * \retval VNCDiscoverySDKErrorFailed If the string can't be used to create the
 * parser.
 *
 * \see VNCDiscoverySDKParser
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKParserGetStringField
 * \see VNCDiscoverySDKParserGetInt32Field
 * \see VNCDiscoverySDKParserGetSubFieldCount
 * \see VNCDiscoverySDKParserDestroy
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKUtilCreateParser(VNCDiscoverySDKInstance *pSdk, const char *string,
    VNCDiscoverySDKParser **ppParser);

/**
 * \brief Gets one string field from the Parser.
 *
 * The field set must be a leaf field, as they are the only ones that are
 * allowed a value. 
 *
 * \param pParser Pointer to the Parser from where the information is retrieved.
 * \param fieldPath The path to the field that contains the value. Must not be
 * NULL. The allowed values for a value are letters (a-z and A-Z),
 * numbers (0-9) and underscore (_). The fields are separated by a dot (.).
 * \param ppResult The location where the result is stored, if the request is
 * successful. The value is a NUL-terminated UTF-8 encoded string. Ownership of
 * the string stays with the Parser. The lifetime of the string is tied to the
 * lifetime of the Parser. If the request fails, this will be NULL.
 *
 * \retval VNCDiscoverySDKErrorNone The field was set successfully.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL,
 * or the fieldPath is not in the expected format.
 * \retval VNCDiscoverySDKErrorNotFound The path does not lead to a field with
 * a value. Please note that the field might exist, but not be a leaf field (so
 * it wouldn't have any value stored).
 *
 * \see VNCDiscoverySDKParser
 * \see VNCDiscoverySDKUtilCreateParser
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKParserGetStringField(VNCDiscoverySDKParser *pParser,
    const char *fieldPath, const char **ppResult);

/**
 * The value of a integer result when a failure occurs. This is not to be used
 * as the guide to determine a failure, it's just defined for clarity.
 *
 * \see VNCDiscoverySDKParserGetInt32Field
 */
#define VNC_DISCOVERY_SDK_UTILS_INT_FAIL_VALUE ((vnc_int32_t)0x80000000)

/**
 * \brief Gets one signed 32 bit integer field from the Parser.
 *
 * The field set must be a leaf field, as they are the only ones that are
 * allowed a value. 
 *
 * \param pParser Pointer to the Parser from where the information is retrieved.
 * \param fieldPath The path to the field that contains the value. Must not be
 * NULL. The allowed values for a value are letters (a-z and A-Z),
 * numbers (0-9) and underscore (_). The fields are separated by a dot (.).
 * \param pResult The location where the result is stored, if the request is
 * successful. If the request fails this will be
 * VNC_DISCOVERY_SDK_UTILS_INT_FAIL_VALUE. Please note that it may be possible
 * to have a request succeeding with the result
 * VNC_DISCOVERY_SDK_UTILS_INT_FAIL_VALUE.
 *
 * \retval VNCDiscoverySDKErrorNone The field was set successfully.
 * \retval VNCDiscoverySDKErrorInvalidParameter If any of the pointers are NULL,
 * or the fieldPath is not in the expected format.
 * \retval VNCDiscoverySDKErrorNotFound The path does not lead to a field with
 * a value. Please note that the field might exist, but not be a leaf field (so
 * it wouldn't have any value stored).
 * \retval VNCDiscoverySDKFailed The field can not be parsed as a 32 bit
 * integer.
 *
 * \see VNCDiscoverySDKParser
 * \see VNCDiscoverySDKUtilCreateParser
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef VNCDiscoverySDKError VNCCALL
VNCDiscoverySDKParserGetInt32Field(VNCDiscoverySDKParser *pParser,
    const char *fieldPath, vnc_int32_t *pResult);

/**
 * \brief Gets the number of "children" of a field.
 *
 * The field set must not be a leaf field.
 *
 * \param pParser Pointer to the Parser from where the information is retrieved.
 * \param fieldPath The path to the field of interest. Must not be NULL. The
 * allowed values for a value are letters (a-z and A-Z), numbers (0-9) and
 * underscore (_). The fields are separated by a dot (.).
 *
 * \return A value greater than 0 if the call is successful, 0 if the field is a
 * leaf field, a negative number indicates that the path is incorrect.
 *
 * \see VNCDiscoverySDKParser
 * \see VNCDiscoverySDKUtilCreateParser
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef vnc_int32_t VNCCALL
VNCDiscoverySDKParserGetSubFieldCount(VNCDiscoverySDKParser *pParser,
    const char *fieldPath);

/**
 * \brief Destroys the Parser.
 *
 * Once the Parser is no longer needed, this must be called to free up any
 * resources.
 *
 * \param pParser The Parser that should be destroyed.
 *
 * \see VNCDiscoverySDKParser
 * \see VNCDiscoverySDKUtilCreateParser
 * \see vncdiscoverysdktypes.h
 * \see VNCDiscoverySDKUtils
 */
typedef void VNCCALL
VNCDiscoverySDKParserDestroy(VNCDiscoverySDKParser *pParser);

/**
 * Structure which contains the utility functions provided by the Discovery SDK.
 *
 * \see VNCDiscoverySDKGetUtils
 * \see VNCDiscovererGetUtils
 * \see VNCDeviceProviderGetUtils
 * \see VNCDiscoverySDKGenerator
 * \see VNCDiscoverySDKParser
 */
typedef struct
{
  /**
   * The Discovery SDK instance that filled in this structure.
   */
  VNCDiscoverySDKInstance *pSDKInstance;
  /**
   * Creates an instance of the generator.
   */
  VNCDiscoverySDKUtilCreateGenerator *createGenerator;
  /**
   * Sets one string field in the Generator.
   */
  VNCDiscoverySDKGeneratorSetStringField *generatorSetStringField;
  /**
   * Sets one signed 32 bit integer field in the Generator.
   */
  VNCDiscoverySDKGeneratorSetInt32Field *generatorSetInt32Field;
  /**
   * Generates the string that corresponds to the structure set within the
   * Generator.
   */
  VNCDiscoverySDKGeneratorGenerate *generatorGenerate;
  /**
   * Destroys the Generator.
   */
  VNCDiscoverySDKGeneratorDestroy *generatorDestroy;
  /**
   * Creates an instance of the Parser.
   */
  VNCDiscoverySDKUtilCreateParser *createParser;
  /**
   * Gets one string field from the Parser.
   */
  VNCDiscoverySDKParserGetStringField *parserGetStringField;
  /**
   * Gets one signed 32 bit integer field from the Parser.
   */
  VNCDiscoverySDKParserGetInt32Field *parserGetInt32Field;
  /**
   * Gets the number of "children" of a field.
   */
  VNCDiscoverySDKParserGetSubFieldCount *parserGetSubFieldCount;
  /**
   * Destroys the Parser.
   */
  VNCDiscoverySDKParserDestroy *parserDestroy;
} VNCDiscoverySDKUtils;

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCDISCOVERYSDKUTILS_H__) */

